within Modelica_LinearSystems2.Math.Matrices.LAPACK;
function dgelqf "Compute LQ factorization of a real matrix A=L*Q"

  input Real A[:,:];
  output Real Aout[size(A, 1),size(A, 2)]=A
    "The lower triangle contains the m-min(m,n) lower trapezoidal matrix L, the elelments above contain the elementary reflectors";
  output Real tau[min(size(A, 1), size(A, 2))]
    "Scalar factors of the elementary reflectors";
  output Integer info;
protected
  Integer m=size(A, 1);
  Integer n=size(A, 2);
  Integer lda=max(1, m);
  Integer lwork=max(1,3*m);
  Real work[lwork];

external "Fortran 77" dgelqf(
    m,
    n,
    Aout,
    lda,
    tau,
    work,
    lwork,
    info) annotation(Library = {"lapack"});

  annotation (Documentation(info="Lapack documentation:

   Purpose
   =======

   DGELQF computes an LQ factorization of a real M-by-N matrix A:
   A = L * Q.

   Arguments
   =========

   M       (input) INTEGER
           The number of rows of the matrix A.  M >= 0.

   N       (input) INTEGER
           The number of columns of the matrix A.  N >= 0.

   A       (input/output) DOUBLE PRECISION array, dimension (LDA,N)
           On entry, the M-by-N matrix A.
           On exit, the elements on and below the diagonal of the array
           contain the m-by-min(m,n) lower trapezoidal matrix L (L is
           lower triangular if m <= n); the elements above the diagonal,
           with the array TAU, represent the orthogonal matrix Q as a
           product of elementary reflectors (see Further Details).

   LDA     (input) INTEGER
           The leading dimension of the array A.  LDA >= max(1,M).

   TAU     (output) DOUBLE PRECISION array, dimension (min(M,N))
           The scalar factors of the elementary reflectors (see Further
           Details).

   WORK    (workspace/output) DOUBLE PRECISION array, dimension (MAX(1,LWORK))
           On exit, if INFO = 0, WORK(1) returns the optimal LWORK.

   LWORK   (input) INTEGER
           The dimension of the array WORK.  LWORK >= max(1,M).
           For optimum performance LWORK >= M*NB, where NB is the
           optimal blocksize.

           If LWORK = -1, then a workspace query is assumed; the routine
           only calculates the optimal size of the WORK array, returns
           this value as the first entry of the WORK array, and no error
           message related to LWORK is issued by XERBLA.

   INFO    (output) INTEGER
           = 0:  successful exit
           < 0:  if INFO = -i, the i-th argument had an illegal value

   Further Details
   ===============

   The matrix Q is represented as a product of elementary reflectors

      Q = H(k) . . . H(2) H(1), where k = min(m,n).

   Each H(i) has the form

      H(i) = I - tau * v * v'

   where tau is a real scalar, and v is a real vector with
   v(1:i-1) = 0 and v(i) = 1; v(i+1:n) is stored on exit in A(i,i+1:n),
   and tau in TAU(i).

   =====================================================================
 "));
end dgelqf;
